#include<algorithm>
#include<iostream>
#include<deque>
#include<list>
#include<vector>
#include<map>
#include<sstream>
#include<iomanip>

#include<cassert>

#include "minisat/minisat/core/Solver.h"
#include "qnodes.hh"
#include "qsolver.hh"

using namespace Minisat;

const int printlitwidth = 8;

std::ostream& operator<<( std::ostream& os, const Lit& l )
{
	std::ostringstream oss;
	oss << (sign(l) ? "~(" : "(") << var( l ) << ")";
	return os << std::setw(printlitwidth) << oss.str();
}


/* 
 * Wir stellen das Tableaux in einem SAT-Solver dar.
 * Die Literale des Solvers repräsentieren die Präfix-Formeln der ML, 
 * jedes solche Literal stellt eine möglich Welt dar.
 * Es wird eine initial / aktuale Welt omega0 erstellt. 
 * Eine Präfix-Formel kodiert über den Präfix die mögliche Welt 
 * und die Formel, die in dieser Welt gelten soll.
 * Ein Präfix-Formel (L,F) besteht aus einem 
 * Solver-Literal L und einer ML-Formel F.
 *
 * Einige der Solver-Literal korrespondieren also mit möglichen Welt 
 * (Präfixe _p),
 * andere regeln nur den Tableaux-Aufbau. (_l) strukturelle Literale
 */


//Pärchen Literal, das die Gruppe kodiert, die Vektoren, die die Elemente kodieren

std::vector<Lit> MLSolver::mkSemilatticeLits() {
	std::vector<Lit> lit;
	for( unsigned int t=0; t < H->primes_count(); t++ ) {
		Lit label = mkLit( solver.newVar() );
		lit.push_back( label );
	}
	return lit;
}

void MLSolver::mkSemilatticeEncoding( const Lit& at, const std::vector<Lit>& lit) {
	for( unsigned int t=0; t < H->primes_count(); t++ ) {
		for( unsigned int s : H->raw[t]) {
			solver.addClause( at, ~lit[t], lit[s] );
		}
	}
}


std::vector<bool> MLSolver::GroupToBoolVec( const Group& grp )
{
	std::vector<bool> g (H->primes_count(), false);
	for( const auto& e : grp )
	{
		g.at(e) = true;
	}
	return g;
}

void MLSolver::mkSemilatticeNonempty( const Lit& at, const std::vector<Lit>& lit ) {
	vec<Lit> nonempty;
	nonempty.push( at );
	for( unsigned int t=0; t < H->primes_count(); t++ ) {
		nonempty.push( lit[t] );
	}
	solver.addClause( nonempty );
}


//check whether a <= b
//that is a[t] -> b[t] for all t
//also, wenn b[t] falsch ist muss auch a[t] falsch sein
bool MLSolver::isLEQ( const std::vector<bool>& a, const std::vector<bool>& b)
{
	assert( a.size() == b.size() );

	for( unsigned int t=0; t < a.size(); t++)
	{
		if( a[t] & !b[t] )
			return false;
	}
	return true;
}

//check whether ~(a <= b)
//es muss also ein a[t] geben, so dass b[t] nicht gilt.
bool MLSolver::isNotLEQ( const std::vector<bool>& a, const std::vector<bool>& b)
{
	assert( a.size() == b.size() );

	for( unsigned int t=0; t < a.size(); t++)
	{
		if( a[t] & !b[t] )
			return true;
	}
	return false;
}

//inserts p -> x <= g into solver
//That is p -> x[t] -> g[t] for all t
//equivalently p -> ~g[t] -> ~x[t] for all t,
//Hence, if g[t] is false, also x[t] has to be false
void MLSolver::mkLEQfromLit( const Lit& p, const std::vector<Lit>& x, const std::vector<bool>& g)
{
	assert( x.size() == g.size() );
	
	for( unsigned int t=0; t < x.size(); t++)
	{
		if( !g[t] )
			solver.addClause( ~p, ~x[t] );
	}
}

// inserts p -> x >= g into solver
//inserts p -> g <= x into solver
//That is p -> (g[t] -> x[t] for all t)
//Hence, if g[t] is true, also x[t] has to be true
void MLSolver::mkLEQfromLit( const Lit& p, const std::vector<bool>& g, const std::vector<Lit>& x)
{
	assert( x.size() == g.size() );
	
	for( unsigned int t=0; t < x.size(); t++)
	{
		if( g[t] )
			solver.addClause( ~p, x[t] );
	}
}

//Experimental!
void MLSolver::mkLEQfromLit( const Lit& p, const std::vector<Lit>& x, const std::vector<Lit>& y)
{
	assert( x.size() == y.size() );
	
	for( unsigned int t=0; t < x.size(); t++)
	{
		solver.addClause( ~p, ~x[t], y[t] );
	}
}

//inserts p -> ~(x <= g) into solver
//Es muss ein t geben, so dass g[t] falsch ist, aber x[t] wahr.
void MLSolver::mkNotLEQfromLit( const Lit& p, const std::vector<Lit>& x, const std::vector<bool>& g)
{
	assert( g.size() == x.size() );
	vec<Lit> tmp;
	for( unsigned int t=0; t < g.size(); t++ ) {
		if( !g[t] )
			tmp.push( x[t] );
	}
	tmp.push( ~p );
	solver.addClause( tmp );
}

//inserts p -> ~(x >= g) into solver i.e. ~( g <= x)!
//Es muss ein t geben, so dass g[t] falsch ist, aber x[t] wahr.
void MLSolver::mkNotLEQfromLit( const Lit& p, const std::vector<bool>& g, const std::vector<Lit>& x)
{
	assert( g.size() == x.size() );
	vec<Lit> tmp;
	for( unsigned int t=0; t < g.size(); t++ ) {
		if( g[t] )
			tmp.push( ~x[t] );
	}
	tmp.push( ~p );
	solver.addClause( tmp );
}


//Pusht eine Clausel lits -> g <= x auf den Solver
void MLSolver::mkNotLEQfromLit( vec<Lit>& lits, const std::vector<bool>& g, const std::vector<Lit>& x)
{
	assert( g.size() == x.size() );
	for( unsigned int t=0; t < g.size(); t++ ) {
		if( g[t] )
			lits.push( ~x[t] );
	}
	solver.addClause( lits );
}


//inserts g <= x -> p into solver
//equivalently ~p -> ~(g <= x)
void MLSolver::mkLEQtoLit( const Lit& p, const std::vector<bool>& g, const std::vector<Lit>& x)
{
	assert( g.size() == x.size() );

	vec<Lit> to_p;
	for( unsigned int t=0; t < g.size(); t++ ) {
		if( g[t] )
			to_p.push( ~x[t] );
	}
	to_p.push( p );
	solver.addClause( to_p );
}


//Encodes p -> g1 <= g2
/*
void MLSolver::mkSemilatticeLEQAt( const Lit p, const std::vector<Lit>& g1, const std::vector<Lit>& g2)
{
	assert( g1.size() == g2.size() );
	
	std::vector<Lit> a (g1.size());
	for( unsigned int t=0; t < g1.size(); t++)
	{
		a.push_back( mkLit( solver.newVar() ) );
	}
	
	for( unsigned int t=0; t < g1.size(); t++)
	{
		solver.addClause(  g1[t],  g2[t],  a[t] ); // ~g1 & ~g2 ->  a
		solver.addClause(  g1[t], ~g2[t],  a[t] ); // ~g1 &  g2 ->  a
		solver.addClause( ~g1[t],  g2[t], ~a[t] ); //  g1 & ~g2 -> ~a
		solver.addClause( ~g1[t], ~g2[t],  a[t] ); //  g1 &  g2 ->  a
	}

	//p -> a
	for( unsigned int t=0; t < g1.size(); t++)
	{
		solver.addClause( ~p, a[t] );
	}
}
*/

/*
// returns g1 <= g2 -> p
void MLSolver::mkSemilatticeLEQ( const std::vector<Lit>& g1, const std::vector<Lit>& g2)
{
	assert( g1.size() == g2.size() );
	Lit p = mkLit( solver.newVar() );
	
	std::vector<Lit> a (g1.size());
	for( unsigned int t=0; t < g1.size(); t++)
	{
		a.push_back( mkLit( solver.newVar() ) );
	}
	
	for( unsigned int t=0; t < g1.size(); t++)
	{
		solver.addClause(  g1[t],  g2[t],  a[t] ); // ~g1 & ~g2 ->  a
		solver.addClause(  g1[t], ~g2[t],  a[t] ); // ~g1 &  g2 ->  a
		solver.addClause( ~g1[t],  g2[t], ~a[t] ); //  g1 & ~g2 -> ~a
		solver.addClause( ~g1[t], ~g2[t],  a[t] ); //  g1 &  g2 ->  a
	}

	vec<Lit> a_to_p;
	for( unsigned int t=0; t < g1.size(); t++)
	{
		a_to_p.push( ~a[t] );
	}
	a_to_p.push( p );
	solver.addClause( a_to_p );

	//p -> a
	for( unsigned int t=0; t < g1.size(); t++)
	{
		solver.addClause( ~p, a[t] );
	}
	return p;
}
*/

// returns a Literal p with p <-> g1 <= g2  / x <= y
Lit MLSolver::mkSemilatticeLEQ( const std::vector<Lit>& g1, const std::vector<Lit>& g2)
{
	assert( g1.size() == g2.size() );
	Lit p = mkLit( solver.newVar() );
	
	std::vector<Lit> a (g1.size());
	for( unsigned int t=0; t < g1.size(); t++)
	{
		a.push_back( mkLit( solver.newVar() ) );
	}
	
	for( unsigned int t=0; t < g1.size(); t++)
	{
		solver.addClause(  g1[t],  g2[t],  a[t] ); // ~g1 & ~g2 ->  a
		solver.addClause(  g1[t], ~g2[t],  a[t] ); // ~g1 &  g2 ->  a
		solver.addClause( ~g1[t],  g2[t], ~a[t] ); //  g1 & ~g2 -> ~a
		solver.addClause( ~g1[t], ~g2[t],  a[t] ); //  g1 &  g2 ->  a
	}

	vec<Lit> a_to_p;
	for( unsigned int t=0; t < g1.size(); t++)
	{
		a_to_p.push( ~a[t] );
	}
	a_to_p.push( p );
	solver.addClause( a_to_p );

	//p -> a
	for( unsigned int t=0; t < g1.size(); t++)
	{
		solver.addClause( ~p, a[t] );
	}
	return p;
}

// returns a Literal p with p <-> g1 <= g2 / g1 <= x
Lit MLSolver::mkSemilatticeLEQ( const std::vector<bool>& g1, const std::vector<Lit>& g2)
{
	assert( g1.size() == g2.size() );
	Lit p = mkLit( solver.newVar() );

	//falls g1, dann muss g2 auch gelten

	vec<Lit> to_p;
	for( unsigned int t=0; t < g1.size(); t++)
	{
		if( g1[t] )
			to_p.push( ~g2[t] );
	}
	to_p.push( p );
	solver.addClause( to_p );
		
	for( unsigned int t=0; t < g1.size(); t++)
	{
		if( g1[t] )
			solver.addClause( ~p, g2[t] );
	}

	return p;
}


// returns a Literal p with p <-> g1 <= g2 / x <= g2
Lit MLSolver::mkSemilatticeLEQ( const std::vector<Lit>& g1, const std::vector<bool>& g2)
{
	assert( g1.size() == g2.size() );
	Lit p = mkLit( solver.newVar() );
	
	//falls nicht g2, dann muss nicht g1 gelten

	vec<Lit> to_p;
	for( unsigned int t=0; t < g1.size(); t++)
	{
		if( !g2[t] )
			to_p.push( g1[t] );
	}
	to_p.push( p );
	solver.addClause( to_p );
		
	for( unsigned int t=0; t < g1.size(); t++)
	{
		if( !g2[t] )
			solver.addClause( ~p, ~g1[t] );
	}
	return p;
}

void MLSolver::makeGroupAtLit( const Lit& group) {
	std::vector<Lit>& lit = poset_lits[ group ];
	for( unsigned int t=0; t < H->primes_count(); t++ ) {
		Lit label = mkLit( solver.newVar() );
		lit.push_back( label );
	}


	for( unsigned int t=0; t < H->primes_count(); t++ ) {
		for( unsigned int s=0; s < H->raw[t].size(); s++) {
			solver.addClause( ~group, ~lit[t], lit[s] );
		}
	}
}

void MLSolver::addMember( const Lit& group, unsigned int m )
{
	assert( poset_lits.find( group ) != poset_lits.end() );
	solver.addClause( ~group, poset_lits[ group ][m] );
}

//makeLit ist ein Wrapper für mkLit.
//Grundsätzlich wird ein neues SoverLiteral erstellt,
//das die Formel pf repräsentiert.
//Der Fall in dem pf ein Literal repräsentiert, bekommt eine Sonderbehandlung.
//In dem Sonderfall könnte man evtl auch processed pflegen.
Lit MLSolver::makeLit( const PrefixedFormula& pf) {
	const world& omega = std::get<0>(pf);
	const value& beta = std::get<1>(pf);  
	const Formula& f = std::get<2>(pf);
	
	//Falls f ein Literal,
	//dann entweder das Atom2Lit zu f zurück,
	//oder ein neues Atom zu Atom2Lit
	//std::cout << __func__ << " on " << "{" << toInt(p) << "}: " << f.print();

	if( f.isLiteral() ) {
		if( f.isBool() ) {
			PrefixedAtom a = std::make_pair( omega, ((BoolNode*)f.getNode())->getAtom() );
			auto it = Atom2Lit.find( a );
			if( it != Atom2Lit.end() )
			{
				//std::cout << " is known Bool Lit " << toInt(it->second) << std::endl;
				return it->second;
			}
			else
			{
				Lit l = mkLit( solver.newVar() );
				Atom2Lit.insert( { a, l } );
				//std::cout << " is unknown Bool Lit " << toInt(l) <<  std::endl;
				return l;
			}
		}
		if( f.isNot() ) {
			PrefixedAtom a = std::make_pair( omega, ((BoolNode*)(f.getNode())->getChild())->getAtom() );
			auto it = Atom2Lit.find(a);
			if( it != Atom2Lit.end() )
			{
				//std::cout << " is known negated Bool Lit " << toInt(~(it->second)) << std::endl;
				return ~(it->second);
			}
			else
			{
				Lit l = mkLit( solver.newVar() );
				Atom2Lit.insert( { a, l } );
				//std::cout << " is unknown negated Bool Lit " << toInt(~l) << std::endl;
				return ~l;
			}
		}
		assert(false);  //never get here
	}
	else {
		Lit l = mkLit( solver.newVar() );
		Lit2Formula.insert( { l, pf } );
		return l;
	}
}


MLSolver::MLSolver( hierarchy* h) : _x("_x") {
	_omega = mkLit( solver.newVar() );  //Initiale Welt
	solver.addClause( _omega );
	H = h;
};

bool MLSolver::update_agenda() {
	agenda.clear();
	
//	std::cout << __func__ << ": START" << std::endl;
	bool sat = solver.solve(assume);
	if( sat )
	{
		//Wir iterieren über sämtliche Lit2Formula
		for( const auto& lit2form : Lit2Formula ) {
			const Lit& l = lit2form.first;
			const PrefixedFormula& pf = lit2form.second;
			
			// Add to agenda, if 
			// * structural true (active in tableaux)
			// * it has not already been processed
			// * and is not atomic
			// Eigentlich können wir doch auch die "isLiteral"s in Processed aufnehmen...
			// Dann bräcuhten wir lediglich eine Set-Difference
			if( solver.modelValue( l ) == l_True)
			{
				if( processed.find( l ) == processed.end() )
				{
					if( !( std::get<2>(pf).isLiteral() ) )
					{
						agenda.insert( l );
					}
				}
			}
		}
		/*
		std::cout << "Model found:\n";
		printFullModel();

		std::cout << "Active model:\n";
		printFullActiveModel();
		
		std::cout << "Clauses:" << std::endl;
		printClauses();

		std::cout << "Processed:" << std::endl;
		printProcessed();

		std::cout << "Agenda:" << std::endl;
		printAgenda();
		

		std::cout << "General Accessibility:" << std::endl;
		printAccessibility();
		std::cout << "Box Formulas:" << std::endl;
		for( const auto& e : BoxFormula ) {
			const world& omega0 = e.first;
			for( const world& omega1 : e.second )
			{
				//Es sollten niemals negierte Literale auftauchen.
				std::cout << (sign(omega0) ? "~(" : "(") << var( omega0 ) << ")" << " -> "
						  << (sign(omega1) ? "~(" : "(") << var( omega1 ) << ")" << std::endl;
			}
		}
		
		std::cout << "Statistics:" << std::endl;
		std::cout << "Number of Clauses: " << solver.nClauses() << std::endl;
		std::cout << "Number of Variables: " << solver.nVars() << std::endl;

		std::cout << "Atom2Lit:" << std::endl;
		for( const auto& ll : Atom2Lit ) {
			const auto& pa = ll.first;
			const world& omega = pa.first;
			const auto& a = pa.second;
			const auto& l = ll.second;
			std::cout << "{" << var(omega) << "}: " <<  a << std::endl;
		}

		std::cout << "World atoms:" << std::endl;
		for( const auto& ll : Atom2Lit ) {
			const auto& pa = ll.first;
			const auto& l = ll.second;
			const world& omega = pa.first;
			const auto& a = pa.second;
			std::cout << "(" << var(omega) << ") |= "
					  << ((solver.modelValue(l) == l_True) ? " " : "~")
					  << a << std::endl;
		}
		

		std::cout << "World Groups:" << std::endl;
		for( const auto& e : poset_lits )
		{
			std::cout << "world {" << var(e.first) << "}: ";
			for( const auto& i : e.second )
				std::cout << var( i ) << " ";
			std::cout <<  std::endl;
		}
		//Zeige die Belegung der Gruppenvariablen an
		std::cout << "Groupvars:" <<std::endl;
		for( const auto& e : poset_lits )
		{
			std::cout << "world {" << var(e.first) << "}: ";
			for( const auto& i : e.second )
				std::cout << "(" << var( i ) << ")"
						  << ((solver.modelValue(i) == l_True) ? " on" : " off")
						  << "\n";
			std::cout <<  std::endl;
		}

		//Zeige alle SubstitutedVars an
		std::cout << "Substitutions" << std::endl;
		for( const auto& e : ForallSubstVarGrpVal )
		{
			std::cout << "From {" << var(e.first) << "}: ";
			for( const auto& i : e.second )
				std::cout << "(ω_" << var( std::get<0>(i) ) << ", β_" << var( std::get<1>(i) ) << ") for " << (std::get<2>(i)).getName() << std::endl;
			std::cout <<  std::endl;
		}
		
		//Zeige die Belegung der Gruppenvariablen an
		std::cout << "Var GroupVars:" <<std::endl;
		for( const auto& e : GrpVar_poset_lits )
		{
			std::cout << "var {" << e.first.second.getName() << " in " << var( e.first.first ) << "}: ";
			for( const auto& i : e.second )
				std::cout << "(" << var( i ) << ")"
						  << ((solver.modelValue(i) == l_True) ? " on" : " off")
						  << "\n";
			std::cout <<  std::endl;
		}
		*/

		
//		std::cout << __func__ << ": END" << std::endl;
		return true;
	}
//	std::cout << __func__ << ": END" << std::endl;	
	return false;
}
	
void MLSolver::add( const Formula& g ) {
	PrefixedFormula f { _omega, _omega, g };
	Lit l = makeLit( f );
	assume.push(l);
	//Das Literal _omega repräsentiert die aktuelle Welt, also (M,omega0) |= \phi
}

bool MLSolver::solve() {
	bool stabilized = false;
	while( !stabilized )
	{
		if( !update_agenda() )
		{
			std::cout << "------Not satisfiable" << std::endl;
			//std::cout << "Clauses:" << std::endl;
			//printClauses();
			return false;
		}
		
		if( agenda.size() == 0 ) {
			std::cout << "------Agenda empty!" << std::endl;
			break;
		}

		
		
		//Process agenda
		//Tripel l,p,f
		stabilized = true;
		
		for( const auto& l0 : agenda )
		{
			//Hole die durch das Solver-Literal repräsentierte PrefixedFormel
			//l0 kodiert (omega0 \models f)
			const PrefixedFormula& pf0 = Lit2Formula.at( l0 );

			
			const world& omega0 = std::get<0>(pf0);
			const value& beta0 = std::get<1>(pf0);
			const Formula& f0 = std::get<2>(pf0);
			
			// l0: ein Solver Literal, das strukturelle Literal
			// f0: Die Formel zum Literal
			// omega0: Der Präfix (selbst ein Literal, das Literal der Welt)
			
			if( f0.isWould() )
			{
				//Simple Rewriting
				PrefixedFormula f1 { omega0, beta0, Ex( _x <= getGroup( f0 ), Dia( _x, getLHS( f0 )) & Box( _x, ~getLHS( f0) | getRHS( f0 ) ) ) };
				Lit l1 = makeLit( f1 );                          //l1 encodes f1
				solver.addClause( ~l0, l1);
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet

			}
			else if( f0.isMight() )
			{
				//Simple Rewriting
				PrefixedFormula f1 { omega0, beta0, Fa( _x <= getGroup( f0 ), ~Dia( _x, getLHS( f0 )) | Dia( _x, getLHS( f0) & getRHS( f0 ) ) ) };
				Lit l1 = makeLit( f1 );                          //l1 encodes f1
				solver.addClause( ~l0, l1);
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet
			}

			else if( f0.isTrueNode() )
			{
				solver.addClause( ~l0, _omega);
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet

			}
			else if( f0.isFalseNode() )
			{
				solver.addClause( ~l0, ~_omega);
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet
			}
			else if( f0.isLEQ() )
			{
				//std::cout << "LEQ Node" << std::endl;
				//Wir suchen nach einer Erfüllbaren Belegung für LEQ!
				const GroupVar& x = getVar(f0);
				const Group& g = getGroup(f0);

                //Ich muss die Gruppe in Literale übersetzen
				if( GrpVar_poset_lits.find( {beta0, x} ) == GrpVar_poset_lits.end() )
				{
					std::cout << "Error in subexpression " << f0.print() << "! "
							  << "The given variable does not occur with in a quanitifer!" << std::endl;
				}
				assert( GrpVar_poset_lits.find( {beta0, x} ) != GrpVar_poset_lits.end() );
				mkLEQfromLit( l0, GrpVar_poset_lits[ {beta0, x} ], GroupToBoolVec( g ) );
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet

			}
			else if( f0.isGEQ() )
			{
				//std::cout << "GEQ Node" << std::endl;
				//Wir suchen nach einer Erfüllbaren Belegung für LEQ!
				const GroupVar& x = getVar(f0);
				const Group& g = getGroup(f0);

                //Ich muss die Gruppe in Literale übersetzen
				if( GrpVar_poset_lits.find( {beta0, x} ) == GrpVar_poset_lits.end() )
				{
					std::cout << "Error in subexpression " << f0.print() << "! "
							  << "The given variable does not occur with in a quanitifer!" << std::endl;
				}
				assert( GrpVar_poset_lits.find( {beta0, x} ) != GrpVar_poset_lits.end() );
				mkLEQfromLit( l0, GroupToBoolVec( g ), GrpVar_poset_lits[ {beta0, x} ] );
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet

			}
			else if( f0.isNotLEQ() )
			{
				//std::cout << "NotLEQ Node" << std::endl;
				//Wir suchen nach einer Erfüllbaren Belegung für LEQ!
				const GroupVar& x = getVar(f0);
				const Group& g = getGroup(f0);

                //Ich muss die Gruppe in Literale übersetzen
				if( GrpVar_poset_lits.find( {beta0, x} ) == GrpVar_poset_lits.end() )
				{
					std::cout << "Error in subexpression " << f0.print() << "! "
							  << "The given variable does not occur with in a quanitifer!" << std::endl;
				}
				assert( GrpVar_poset_lits.find( {beta0, x} ) != GrpVar_poset_lits.end() );
				mkNotLEQfromLit( l0, GrpVar_poset_lits[ {beta0, x} ], GroupToBoolVec( g ) );
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet

			}
			else if( f0.isNotGEQ() )
			{
				//std::cout << "NotGEQ Node" << std::endl;
				//Wir suchen nach einer Erfüllbaren Belegung für LEQ!
				const GroupVar& x = getVar(f0);
				const Group& g = getGroup(f0);

                //Ich muss die Gruppe in Literale übersetzen
				if( GrpVar_poset_lits.find( {beta0, x} ) == GrpVar_poset_lits.end() )
				{
					std::cout << "Error in subexpression " << f0.print() << "! "
							  << "The given variable does not occur with in a quanitifer!" << std::endl;
				}
				assert( GrpVar_poset_lits.find( {beta0, x} ) != GrpVar_poset_lits.end() );
				mkNotLEQfromLit( l0, GroupToBoolVec( g ), GrpVar_poset_lits[ {beta0, x} ] );
				
				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet

			}

			// alpha rule
			else if( f0.isAnd() )
			{
				PrefixedFormula f1 { omega0, beta0, getLHS( f0 ) };  //f1 := (omega0 |= LHS(f0) ) 
				PrefixedFormula f2 { omega0, beta0, getRHS( f0 ) };  //f2 := (omega0 |= RHS(f0) )
				Lit l1 = makeLit( f1 );                          //l1 encodes f1
				Lit l2 = makeLit( f2 );                          //l2 encodes f2
				
				solver.addClause( ~l0, l1 );                     //l0 -> l1
				solver.addClause( ~l0, l2 );                     //l0 -> l2
				
 				//Auxillary structures
				processed.insert( l0 );                          //worked off l0
				stabilized = false;                              //not stable yet
			}

			//beta rule
			else if( f0.isOr() )
			{
				PrefixedFormula f1 { omega0, beta0, getLHS( f0 ) };  //f1 := (omega0 |= LHS(f0) ) 
				PrefixedFormula f2 { omega0, beta0, getRHS( f0 ) };  //f2 := (omega0 |= LHS(f0) ) 
				Lit l1 = makeLit( f1 );                         //l1 encodes f1
				Lit l2 = makeLit( f2 );                         //l2 encodes f2
				
				solver.addClause( ~l0, l1, l2 );                //l0 -> l1 or l2

				//Auxillary structures
				processed.insert( l0 );                         //worked off l0
				stabilized = false;                             //not stable yet
			}

			//nu rule ohne quantor
			else if( f0.isDia() )
			{
				const world& omega1 = l0;                         //omega1 = l0
				//Wir vergessen die Belegungen!
				PrefixedFormula f1 { omega1, omega1, getChild( f0 ) }; //f1 := (omega1 |= f0)  
				Lit l1 = makeLit( f1 );                           //l1 encodes f1

				//Wir erstellen genügend viele neue Literal, speichern den Vektor dieser Literale in poset_lits[ omega1 ].
				//Außerdem fügen wir die Formel "l0 -> P" zum Solver hinzu, wobei
				//P besagt, dass die Literale in poset_lits[ omega1 ] die Ordnungsstruktur vom Poset respektieren

				std::vector<Lit> lit = mkSemilatticeLits();  //Baue Literale für ein Encoding des Semilattices [p1,p2,...,pn]
				poset_lits[ omega1 ] = lit;                  //poset_lits[ omega1 ] speichert die Belegung für omega1
				mkSemilatticeEncoding( ~l0, lit);            //in l0 gilt die gruppe

				//Encodes DiaGroup <= literals of omega
				mkLEQfromLit( l0, GroupToBoolVec( getGroup(f0) ), lit);
				
				//Hier möchte ich auch ein solver.addClause( ~l0, g)
				//for( const auto& e : getGroup( f0 ).members ) {
				//	solver.addClause( ~l0, lit[e] );
				//}

				solver.addClause( ~l0, l1 );                      //l0 -> l1 

				//Auxillary structures
				Access[ omega0 ].insert( omega1 );                //omega0 -> omega1
				processed.insert( l0 );                           //worked off l0
				stabilized = false;                               //not stable yet
			}

			//nu regel mit quantor
			else if( f0.isVarDia() )
			{
				GroupVar x = getVar( f0 );

				//Das strukturelle Literal wird zu einem Welt-Literal!
				const world& omega1 = l0;
				PrefixedFormula f1 { omega1, omega1, getChild( f0 ) }; //f1 = (omega1 |= f0)
				Lit l1 = makeLit( f1 );

				//if formula is not well formed an assertion fails
				assert( GrpVar_poset_lits.find( { beta0, x} ) != GrpVar_poset_lits.end() );

				//Wir kopieren einfach die var_poset_lits an die Stelle von poset_lits[ omega1 ]
				poset_lits[ omega1 ] = GrpVar_poset_lits[ {beta0, x} ];  //alt

				/*
				//Experimental!
				std::vector<Lit> lit = mkSemilatticeLits();  //Baue Literale für ein Encoding des Semilattices [p1,p2,...,pn]
				poset_lits[ omega1 ] = lit;                  //poset_lits[ omega1 ] speichert die Belegung für omega1
				mkSemilatticeEncoding( ~l0, lit); //in l0 gilt die gruppe

				mkLEQfromLit( l0, GrpVar_poset_lits[ {beta0, x} ], lit); //neu
				
				//Und wenn wir nur die Bedingung x <= omega & ... kodieren? vgl oben.
				
				//Damit werden alle boxen, die hier reinschreiben dürfen automatisch die entsprechenden Literale verwenden
				//Ich könnte auch eine Größere Menge erzeugen, würde dass sinn machen? Weiß ich nicht!
				*/
				
				solver.addClause( ~l0, l1 );
				//muss jetzt auch die Klausel gelten, die diese Element kodiert, oder folgt das implizit? // gilt implizit.

				//Auxillary structures
				Access[ omega0 ].insert( omega1 );                   //omega0 -> omega1
				processed.insert( l0 );                              //worked off l0
				stabilized = false;                                  //not stable yet
			}
			
			else if( f0.isBox() )
			{
                //Falls diese Box noch nicht bekannt ist, sollten wir sie hinzufügen.
				BoxFormula[ omega0 ].insert( l0 );                   //l0 ist eine Box unterhalb von omega0
				
                //Besorge alle aktiven möglichen Welten, die für die Box accessible sind.
				//und wende die Formel in diesen Welten an.
				//Hole alle Welten, die verbunden sind
				//Wir filtern auf alle erreichbaren aktiven Welten, die durch die Gruppe erreicht werden können, und auch nicht in BoxAccess liegen

				const std::set<world>& toaccess = Access[ omega0 ];        //prinzipiell erreichbar
				const std::set<world>& accessed = BoxAccessed[ l0 ]; //schon besucht
				
				std::vector<world> tmpOmega1;
				std::set_difference( toaccess.begin(), toaccess.end(), accessed.begin(), accessed.end(), std::back_inserter(tmpOmega1) );
				
				//Nun müssen wir noch alle Welten entfernen, die inaktiv sind
				
				std::vector<world> Omega1;
					
				for( const world& omega1 : tmpOmega1 )
				{
					if( solver.modelValue(omega1) == l_True)
					{
						Omega1.push_back( omega1 );
					}
				}

				//In Omega1 sind alle Welten, die ich noch besuchen muss
				
				const Group& box_grp = getGroup( f0 );

				for( const world& omega1 : Omega1 ) {
					PrefixedFormula diaf = Lit2Formula.at( omega1 );
					//ich schreibe die Formel omega1, omega1 |= f0 raus. Wie sieht das zweite Argument aus?
					//Das muss doch das beta von der anderen formel sein? Oder? Spielt es evtl. gar keine Rolle?
					//omega1, omega1, weil ich am Eingang alle Belegungen vergessen habe!
					PrefixedFormula f1 { omega1, omega1, getChild( f0 ) };
					Lit l1 = makeLit( f1 );

					//Das sollte auf jeden Fall vorhanden sein!
					assert( poset_lits.find( omega1 ) != poset_lits.end() );
					std::vector<Lit>& lit = poset_lits[ omega1 ];

					// l0 -> (Wenn a <= lit, dann l1)
					// l0 -> ( a <= lit -> l1)
					// l0 & a <= Lit -> l1
					// ~l1 -> ~(l0 & a <= lit)
					// l1 oder ~l0 oder ~(a<=lit)
					// ~l0 oder l1 oder ~a<=lit
					//Experimental (partially ugly)
					vec<Lit> tmp;
					tmp.push( ~l0 );
					tmp.push( l1 );
					mkNotLEQfromLit( tmp, GroupToBoolVec( box_grp ), lit );
					// l0 & ~l1 -> ~ (a <= lit)

					/*
					// l0 -> ( lit(e) -> l1 )
					// <=> l0 & lit(e) -> l1
					//Wir kodieren das Poset (l0 & lit(e)) -> l1
					//Falls also die Box-Gruppe im Encoding von omega1 wahr ist
					vec<Lit> groupencode;
					groupencode.push( ~l0 );
					for( const auto& e : box_grp.members ) {
						groupencode.push( ~lit[e] );
					}
					groupencode.push( l1 );
					solver.addClause( groupencode );
					
					std::cout << "Gruppe in Box kodiert" << std::endl;
					*/
					
					BoxAccessed[ l0 ].insert( omega1 );
					stabilized = false;
					
				}
			}

			else if( f0.isVarBox() )
			{
                //Falls diese Box noch nicht bekannt ist, sollten wir sie hinzufügen.
				BoxFormula[ omega0 ].insert( l0 );                   //l0 ist eine Box unterhalb von omega0

				//Besorge alle aktiven möglichen Welten, die für die Box accessible sind.
				//und wende die Formel in diesen Welten an.
				//Hole alle Welten, die verbunden sind
				//Wir filtern auf alle erreichbaren aktiven Welten, die durch die Gruppe erreicht werden können, und auch nicht in BoxAccess liegen

				const std::set<world>& toaccess = Access[ omega0 ];        //prinzipiell erreichbar
				const std::set<world>& accessed = BoxAccessed[ l0 ]; //schon besucht

				std::vector<world> tmpOmega1;
				std::set_difference( toaccess.begin(), toaccess.end(), accessed.begin(), accessed.end(), std::back_inserter(tmpOmega1) );
				
				//Nun müssen wir noch alle Welten entfernen, die inaktiv sind
				
				std::vector<world> Omega1;
					
				for( const world& omega1 : tmpOmega1 )
				{
					if( solver.modelValue(omega1) == l_True)
					{
						Omega1.push_back( omega1 );
					}
				}

				//In Omega1 sind alle Welten, die ich noch besuchen muss
				
				const GroupVar x = getVar( f0 );

				assert( GrpVar_poset_lits.find( {beta0, x} ) != GrpVar_poset_lits.end() );
				const std::vector<Lit>& boxlit = GrpVar_poset_lits[ {beta0, x} ];
					
				for( const world& omega1 : Omega1 ) {
					PrefixedFormula f1 { omega1, omega1, getChild( f0 ) };
					Lit l1 = makeLit( f1 );

					//Wir holen uns die Kodierung von omega1
										
					//Das sollte auf jeden Fall vorhanden sein!
					assert( poset_lits.find( omega1 ) != poset_lits.end() );
					std::vector<Lit>& lit = poset_lits[ omega1 ];
				   
					//Falls also die Box-Gruppe im Encoding von omega1 wahr ist
					//Faktisch also: Falls die Primelemente von X auch Primelemente von omega1 sind

                    //Wir kodieren die Äquivalenz von boxlit und lit
					//als eine Bedingung
					// 
					// (boxlit[1] -> lit[1]) <-> a[1]

					std::vector<Lit> lit_a;
					for( unsigned int t=0; t < H->primes_count(); t++ )
					{
						lit_a.push_back( mkLit( solver.newVar() ) );
					}

					for( unsigned int t=0; t < H->primes_count(); t++ )
					{
						solver.addClause(  boxlit[t],  lit[t],  lit_a[t] ); // ~b & ~l ->  a
						solver.addClause(  boxlit[t], ~lit[t],  lit_a[t] ); // ~b &  l ->  a
						solver.addClause( ~boxlit[t],  lit[t], ~lit_a[t] ); //  b & ~l -> ~a
						solver.addClause( ~boxlit[t], ~lit[t],  lit_a[t] ); //  b &  l ->  a
					}

					// l0 -> ( /\ lit_a(e) -> l1 )
					// <=> l0 & /\ lit_a(e) -> l1
					// <=> ~l0 | \/ ~lit_a(e) | l1
					vec<Lit> encode;
					encode.push( ~l0 );
					for( unsigned int t=0; t < H->primes_count(); t++ )
					{
						encode.push( ~lit_a[t] );
					}
					encode.push( l1 );
					solver.addClause( encode );
					
					BoxAccessed[ l0 ].insert( omega1 );
					stabilized = false;
				}
			}

			else if( f0.isExists() )
			{
				const Group& lb = getLowerBound( f0 );
				const GroupVar& x = getVar( f0 );
				const Group& ub = getUpperBound( f0 );

				if( !leq( lb, ub ) )
				{
					std::cout << "quantification over empty subsets is not yet supported!" << std::endl;
					assert(false);
				}
					
				//Neu: Wir führen eine neue Belegung für die GroupVar ein
				const value& beta1 = l0;
				std::vector<Lit> x_lits = mkSemilatticeLits();
				GrpVar_poset_lits[ {beta1, x} ] = x_lits;
				mkSemilatticeEncoding( ~l0, x_lits);
				mkSemilatticeNonempty( ~l0, x_lits);

				Lit l1;
				if( lb == Group{} )
				{
					PrefixedFormula f1 { omega0, beta1, LEQ(x,ub) & getChild( f0 ) };
					l1 = makeLit( f1 );
				}
				else
				{
					PrefixedFormula f1 { omega0, beta1, LEQ(lb,x) & LEQ(x,ub) & getChild( f0 ) };
					l1 = makeLit( f1 );
				}

				solver.addClause( ~l0, l1 );
				processed.insert( l0 );  //l0 is complete processed
				stabilized = false;      //not stable yet
			}

			else if( f0.isForall() )
			{
				const Group& lb0 = getLowerBound( f0 );				
				const GroupVar& x0 = getVar( f0 );
				const Group& ub0 = getUpperBound( f0 );

				if( !leq( lb0, ub0 ) )
				{
					std::cout << "quantification over empty subsets is not yet supported!" << std::endl;
					assert(false);
				}

				//Only upper bound has to be substiuted
				std::set<Group> candidateSubstitute;
				candidateSubstitute.insert( ub0 );

				std::set<varvaluation> candidateSubstituteVar2;
				std::set<varvaluation>& substitutedVar2 = ForallSubstVarGrpVal[ l0 ];
				
				//wir durchlaufen alle aktive, erreichbaren Dia-Formeln
				for( const Lit& l : Access[ omega0 ] )
				{
					if( solver.modelValue( l ) != l_True )
						continue;

					const PrefixedFormula& pf = Lit2Formula.at( l );
					const Formula& f = std::get<2>(pf);

					//Collect all formulas whose group satisfies the bound condition
					if( f.isDia() )
					{
						const Group& g = getGroup( f );

						if( leq(lb0, g) && leq( g, ub0 ) ) //HERE
							candidateSubstitute.insert( g );
						continue;
					}

					if( f.isVarDia() )
					{
						const world& omega = std::get<0>(pf);
						const value& beta = std::get<1>(pf);
						const GroupVar& x = getVar( f );   //fishy!
						if( substitutedVar2.find( {omega,beta,x} ) == substitutedVar2.end() )
						{
							candidateSubstituteVar2.insert( {omega,beta,x} );
						}
							
						continue;
					}

					assert(false);
				}

				//wir durchlaufen alle aktive, erreichbaren Box-Formeln
				for( const Lit& l : BoxFormula[ omega0 ] )
				{
					if( solver.modelValue( l ) != l_True )
						continue;

					const PrefixedFormula& pf = Lit2Formula.at( l );
					const Formula& f = std::get<2>(pf);

					//Collect all formulas whose group satisfies the bound condition
					if( f.isBox() )
					{
						const Group& g = getGroup( f );
						if( leq( g, lb0 ) && leq( g, ub0 ) )
							candidateSubstitute.insert( g );
						continue;
					}

					if( f.isVarBox() )
					{
						const world& omega = std::get<0>(pf);
						const value& beta = std::get<1>(pf);
						const GroupVar& x = getVar( f );   //fishy!
						if( substitutedVar2.find( {omega,beta,x} ) == substitutedVar2.end() )
						{
							candidateSubstituteVar2.insert( {omega,beta,x} );
						}

						continue;
					}
					
					assert(false);
				}

				std::set<Group>& isSubstituted = ForallSubstitutionGroups[ l0 ];
				std::set<Group> toSubstitute;
				std::set_difference( candidateSubstitute.begin(), candidateSubstitute.end(),
									 isSubstituted.begin(), isSubstituted.end(),
									 std::inserter(toSubstitute, toSubstitute.end()));

				
				//substitution for fixed groups
				for( const auto& g : toSubstitute )
				{
					PrefixedFormula f1 { omega0, omega0, getChild( f0 ).subst( g, x0 ) };
					Lit l1 = makeLit( f1 );
					solver.addClause( ~l0, l1 );

					isSubstituted.insert( g );
					stabilized = false;

				}

				//substitutions for variable groups
				for( const varvaluation& v : candidateSubstituteVar2 )
				{
					const world& omega = std::get<0>(v);
					const value& beta = std::get<1>(v);
					const GroupVar& x = std::get<2>(v);

					Lit l1;
					if( lb0 == Group{} )
					{
						PrefixedFormula f1 { omega, beta, NotLEQ( x, ub0 ) | getChild( f0 ).subst( x, x0) };
						l1 = makeLit( f1 );
					}
					else
					{
						PrefixedFormula f1 { omega, beta, NotLEQ(lb0, x) | NotLEQ( x, ub0 ) | getChild( f0 ).subst( x, x0) };
						l1 = makeLit( f1 );
					}

					solver.addClause( ~l0, l1 );
						
					substitutedVar2.insert( {omega, beta, x } );
					stabilized = false;
				}

			}

			
			else
			{
				std::cout << "UNKNOWN!" << std::endl;
				assert(false);
			}
		}
		
	}
	return true;
}

void MLSolver::printModel(const Formula& f) {
	std::cout << "Model for " << f.print() << std::endl;
	/*
	std::cout << "Atom2Lit" << std::endl;
	for( const auto& ll : Atom2Lit ) {
		const auto& pa = ll.first;
		const auto& omega = pa.first;
		const auto& a = pa.second;
		const auto& l = ll.second;
		std::cout << "{" << var(omega) << "}: " <<  a << " ... ";
		std::cout << (sign( ll.second ) ? "~(" : "(") << var( ll.second ) << ") ... ";
		if( solver.modelValue( ll.second ) == l_True)
			std::cout << "true .... ";
		else
			std::cout << "false .... ";
		if( solver.modelValue( omega ) == l_True)
			std::cout << "the world is true" << std::endl;
		else
			std::cout << "the world is false" << std::endl;
	}
	*/
	std::cout << "Model Accessibility: " << std::endl;
	printActiveAccessibility();

	//Jetzt folgt eine Angabe der Welten, wir nehmen nur die aktiven.

	//Atom2Lit ist map< pair< world, Atom>, lit>
	//geordnet strikt weak order, also gibt es ggf Elemente, die ungleich aber äquivalent sind

	Lit last_world = _omega;
	std::cout << "(ω_" << var(last_world) << ") belongs to ground truth.\n"
			  << "(ω_" << var(last_world) << ") ⊧ ";
	
	for( const auto& ll : Atom2Lit ) {

		const auto& pa = ll.first;    //prefixed atom
		const auto& l = ll.second;    //literal
		const auto& omega = pa.first; //Die Welt
		const auto& a = pa.second;    //Das Atom

		if( solver.modelValue( omega ) != l_True )
			continue;

		if( last_world != omega )
		{
			std::cout << std::endl
					  << "(ω_" << var(omega) << ") belongs to ";
			assert( poset_lits.find(omega) != poset_lits.end() );
			std::vector<Lit> lit = poset_lits[ omega ];
			std::cout << "(";
			for( const auto& i : lit )
				std::cout << ((solver.modelValue(i) == l_True) ? "1" : "0");
			std::cout << ")." << std::endl;
			std::cout << "(ω_" << var(omega) << ") ⊧ ";
		}
		
		last_world = omega;
		std::cout << ((solver.modelValue( l ) == l_True ) ? " " : "~") << a << ", ";
	}
	std::cout << std::endl;

	std::cout << "Variables:" <<std::endl;
	for( const auto& e : GrpVar_poset_lits )
	{
		std::cout << e.first.second.getName() << " in ";
		std::cout << Lit2Formula.at( e.first.first ) << " " << ((solver.modelValue(e.first.first) == l_True) ? " active" : " inactive") << std::endl;
		std::cout << e.first.second.getName() << " belongs to ";
		std::cout << "(";
		for( const auto& i : e.second )
			std::cout << ((solver.modelValue(i) == l_True) ? "1" : "0");
		std::cout << ")." << std::endl;
	}

}

void MLSolver::printFullModel() {
	//Wir iterieren über sämtliche Lit2Formula
	for( const auto& lit2form : Lit2Formula ) {
		const auto& l = lit2form.first;
		const auto& pf = lit2form.second;
			
		std::cout << l << " " << ((solver.modelValue(l) == l_True) ? "   active" : " inactive")
				  << ": " << pf << "\n";
		
	}
	//Wir holen jetzt noch alle Atom2Lit hinzu
	for( const auto& ll : Atom2Lit )
	{
		const auto& pa = ll.first;
		const auto& l = ll.second;
		const world& omega = pa.first;
		const auto& a = pa.second;

		std::cout << l 
				  << "   is atom"
				  << ": "
				  << "(ω_" << var(omega) << ") ⊧ "
			      << ((solver.modelValue(l) == l_True) ? "" : "~")
				  << a
				  << std::endl;
	}
		


}

void MLSolver::printFullActiveModel() {
	//Wir iterieren über sämtliche Lit2Formula
	for( const auto& lit2form : Lit2Formula )
	{
		const auto& l = lit2form.first;
		const auto& pf = lit2form.second;
			
		if( solver.modelValue(l) == l_True)
		{
			std::cout << l << ": " << pf << "\n";
		}
	}
}

void MLSolver::printClauses() {
	for( ClauseIterator cit = solver.clausesBegin();
		 cit != solver.clausesEnd(); ++cit )
	{
		for( int i=0; i < (*cit).size(); i++)
		{
			std::cout << ( sign( (*cit)[i] )  ? "~" : "" ) << "(" << var( (*cit)[i] ) << ") ";
		}
		std::cout << std::endl;
	}
}

void MLSolver::printAgenda() {
	for( const auto& l : agenda ) {
		const auto& it = Lit2Formula.find(l);
		assert( it != Lit2Formula.end() );
		const auto& ll = it->first;
		const auto& pf = it->second;
		std::cout << ll << ": " << pf << "\n";
	}
}

void MLSolver::printProcessed() {
	for( const auto& l : processed ) {
		const auto& it = Lit2Formula.find(l);
		assert( it != Lit2Formula.end() );
		const auto& ll = it->first;
		const auto& pf = it->second;
		std::cout << ll << ": " << pf << "\n";
	}
}

void MLSolver::printAccessibility() {
	for( const auto& e : Access ) {
		const world& omega0 = e.first;
		for( const world& omega1 : e.second )
		{
			//Es sollten niemals negierte Literale auftauchen.
			std::cout << "(ω_" << var(omega0) << ")"
					  << " ↦ "
					  << "(ω_" << var( omega1 ) << ")"
					  << std::endl;
		}
	}
}

void MLSolver::printActiveAccessibility() {
	for( const auto& e : Access ) {
		const world& omega0 = e.first;
		if( (solver.modelValue(omega0) != l_True ) )
			continue;
		for( const world& omega1 : e.second )
		{
			if( (solver.modelValue(omega1) == l_True ) )
				std::cout << "(ω_" << var(omega0) << ")"
						  << " ↦ "
						  << "(ω_" << var( omega1 ) << ")"
						  << std::endl;
		}
	}

}
