#ifndef NODES_HH
#define NODES_HH

#include<tuple>
#include<vector>
#include<map>
#include<set>
#include<string>
#include<iostream>

//Groups are sorted vectors
typedef std::vector<unsigned int> GroupBase;

class Group : public GroupBase
{
public:
	Group( const std::initializer_list<unsigned int>& list ) : GroupBase( list ) {
		std::sort( this->begin(), this->end() );
	};
};

inline bool leq(const Group& a, const Group& b) {
	return std::includes( b.begin(), b.end(), a.begin(), a.end() );
}

std::vector<bool> GroupToBoolVec( const Group& g );
std::ostream& operator<<( std::ostream& os, const Group& g );

typedef std::vector< std::vector<unsigned int> > Poset;

class hierarchy {
public:
	hierarchy( const Poset& P ) : raw( P ) {};

	inline Poset::size_type primes_count() { return raw.size(); }

//private:
	Poset raw;
};


class GroupVar
{
public:
    GroupVar() : name( "" ) {};
    GroupVar( const char* a) : name( a ) {};
    GroupVar( const std::string& a ) : name( a ) {};

    inline bool operator==( const GroupVar& other ) const {
        return (name == other.name);
    }

    inline bool operator<( const GroupVar& other ) const {
        return (name < other.name);
    }
	inline std::string getName() const { return name; }
	
private:
    std::string name;
};

typedef std::tuple<Group, GroupVar, Group> qterm;
inline qterm operator<=( const GroupVar& x, const Group& ub) {
	return { {}, x, ub };
}
inline qterm operator<=( const Group& lb, const GroupVar& x) {
	return { lb, x, {} };
}
inline qterm operator<=( const qterm& q, const Group& ub) {
	return { std::get<0>(q), std::get<1>(q), ub };
}

/*
see https://stackoverflow.com/questions/11420448/initializer-lists-and-rhs-of-operators
inline qterm operator<=( const GroupVar& x, std::initializer_list<unsigned int>& list) {
	std::vector<unsigned int> g(list.size());
	std::copy( list.begin(), list.end(), g.begin());
	return { x, g };
}
*/

class Atom
{
    friend std::ostream& operator<<( std::ostream& os, const Atom& a );

public:
    Atom() : name( "" ) {};
    Atom( const char* a) : name( a ) {};
    Atom( const std::string& a ) : name( a ) {};

    inline bool operator==( const Atom& other ) const {
        return (name == other.name);
    }

    inline bool operator<( const Atom& other ) const {
        return (name < other.name);
    }

	inline std::string getName() const { return name; }
private:
    std::string name;
};


inline std::ostream& operator<<( std::ostream& os, const Atom& a )
{
    return os << a.name;
}


class Formula;

class Node
{
	friend class Formula;
protected:
    Node() {};
public:
    virtual std::string print() = 0;
    virtual Node* getNode() { return this; };  //virtual?
	virtual Group getGroup() { assert(false); };
	virtual GroupVar getVar() { assert(false); };
	virtual Node* getChild() { assert(false); };
	virtual Node* getRhsNode() { assert(false); };
	virtual Node* getLhsNode() { assert(false); };
};

//Leaf-Nodes
class TrueNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
protected:
	TrueNode() {};
public:
	std::string print();
};

class FalseNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
protected:
	FalseNode() {};
public:
	std::string print();
};

class BoolNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
protected:
    BoolNode( const Atom& s ) : content( s ) {};	
public:
    std::string print();

	inline Atom getAtom() { return content; };
private:
    Atom content;
};

//x <= g
class LEQNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula LEQ( const GroupVar& x, const Group& g );
protected:
    LEQNode( const GroupVar& x, const Group& g ) : lhs(x), rhs(g) {};
public:
    std::string print();
    GroupVar getVar() { return lhs; }
    Group getGroup() {return rhs; }
private:
	GroupVar lhs;
    Group rhs;
};

//x >= g
class GEQNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula GEQ( const GroupVar& x, const Group& g );
	friend Formula LEQ( const Group& g, const GroupVar& x );
protected:
    GEQNode( const GroupVar& x, const Group& g ) : lhs(x), rhs(g) {};
public:
    std::string print();
    GroupVar getVar() { return lhs; }
    Group getGroup() {return rhs; }
private:
	GroupVar lhs;
    Group rhs;
};

//~(x <= g)
class NotLEQNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula NotLEQ( const GroupVar& x, const Group& g );
protected:
    NotLEQNode( const GroupVar& x, const Group& g ) : lhs(x), rhs(g) {};
public:
    std::string print();
    GroupVar getVar() { return lhs; }
    Group getGroup() {return rhs; }
private:
	GroupVar lhs;
    Group rhs;
};

//~(x >= g)
class NotGEQNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula NotGEQ( const GroupVar& x, const Group& g );
	friend Formula NotLEQ( const Group& g, const GroupVar& x );
protected:
	NotGEQNode( const GroupVar& x, const Group& g ) : lhs(x), rhs(g) {};
public:
    std::string print();
    GroupVar getVar() { return lhs; }
    Group getGroup() {return rhs; }
private:
	GroupVar lhs;
    Group rhs;
};


//Parent-Nodes, we have single child and two children
class NegatedNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
protected:
    NegatedNode( Node* n ) : bodyNode( n ) {};
public:
    std::string print();
    Node* getChild() { return bodyNode ; }
private:
    Node* bodyNode;
};


class AndNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula operator&(const Formula& lhs, const Formula& rhs );
protected:
    AndNode( Node* lhs, Node* rhs ) : lhsNode( lhs ), rhsNode( rhs ) {};
public:
    std::string print();
	Node* getLhsNode() { return lhsNode; }
    Node* getRhsNode() { return rhsNode; }
private:
    Node* lhsNode;
    Node* rhsNode;
};


class OrNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula operator|(const Formula& lhs, const Formula& rhs );
protected:
    OrNode( Node* lhs, Node* rhs ) : lhsNode( lhs ), rhsNode( rhs ) {};
public:
	std::string print();
    Node* getLhsNode() { return lhsNode; }
    Node* getRhsNode() { return rhsNode; }
private:
    Node* lhsNode;
    Node* rhsNode;
};

class DiamondNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Dia(const Group& g, const Formula& lhs);
protected:
    DiamondNode( Node* node, const Group& g ) : bodyNode(node), who(g) {};
public:
    std::string print();
	Node* getChild() { return bodyNode; }
	Group getGroup() { return who; }
	
private:
    Node* bodyNode;
    Group who;
};

class BoxNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Box(const Group& g, const Formula& lhs);
protected:
    BoxNode( Node* node, const Group& g ) : bodyNode(node), who(g) {};
public:
    std::string print();
    Node* getChild() { return bodyNode ; }
	Group getGroup() { return who; }
private:
    Node* bodyNode;
    Group who;
};

class DiamondVarNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Dia(const GroupVar& x, const Formula& lhs);
protected:
	DiamondVarNode( Node* node, const GroupVar& varname ) : bodyNode(node), var(varname) {};
public:
    std::string print();
	Node* getChild() { return bodyNode; }
	GroupVar getVar() { return var; }
private:
    Node* bodyNode;
	GroupVar var;
};

class BoxVarNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Box(const GroupVar& x, const Formula& lhs);
protected:
    BoxVarNode( Node* node, const GroupVar& varname ) : bodyNode(node), var(varname) {};
public:
    std::string print();
    Node* getChild() { return bodyNode ; }
	GroupVar getVar() { return var; }
private:
    Node* bodyNode;
	GroupVar var;
};

// Stellt eine <= Restriction der Gruppe da
class ExistsNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Ex(const qterm& condition, const Formula& lhs);
protected:
	ExistsNode( Node* node, const Group& lb, const GroupVar& x, const Group& ub ) : bodyNode(node), lowerbound( lb ), var( x ), upperbound( ub ) {};
public:
	std::string print();
    Node* getChild() { return bodyNode; };
	GroupVar getVar() { return var; };
//	Group getBound() { return upperbound; };
	Group getUpperBound() { return upperbound; };
	Group getLowerBound() { return lowerbound; };
private:
    Node* bodyNode;
	Group lowerbound; 
	GroupVar var;
	Group upperbound;
};

// Stellt eine <= Restriction der Gruppe da
class ForallNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Fa(const qterm& condition, const Formula& lhs);
protected:
	ForallNode( Node* node, const Group& lb, const GroupVar& x, const Group& ub ) : bodyNode(node), lowerbound( lb), var( x ), upperbound( ub ) {};
public:
	std::string print();

    Node* getChild() { return bodyNode; }
	GroupVar getVar() { return var; };
//	Group getBound() { return upperbound; };
	Group getUpperBound() { return upperbound; };
	Group getLowerBound() { return lowerbound; };
private:
    Node* bodyNode;
	Group lowerbound; 
	GroupVar var;
	Group upperbound;
};


class WouldNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Would(const Group& g, const Formula& lhs, const Formula& rhs);
protected:
    WouldNode( Node* lhs, Node* rhs, const Group& g ) : lhsNode( lhs ), rhsNode( rhs ), who( g ) {};
public:
    std::string print();
    Node* getLhsNode() { return lhsNode; }
    Node* getRhsNode() { return rhsNode; }
	Group getGroup() { return who; }
private:
    Node* lhsNode;
    Node* rhsNode;
    Group who;
};

class MightNode : public Node
{
	friend class Formula;
	friend Formula operator~(const Formula& lhs);
	friend Formula Might(const Group& g, const Formula& lhs, const Formula& rhs);
protected:
    MightNode( Node* lhs, Node* rhs, const Group& g ) : lhsNode( lhs ), rhsNode( rhs ), who( g ) {};
public:
    std::string print();
    Node* getLhsNode() { return lhsNode; }
    Node* getRhsNode() { return rhsNode; }
	Group getGroup() { return who; }
private:
    Node* lhsNode;
    Node* rhsNode;
    Group who;
};


class Formula
{
	friend Formula operator~(const Formula& lhs);
    friend Formula operator&(const Formula& lhs, const Formula& rhs );
    friend Formula operator|(const Formula& lhs, const Formula& rhs );
	
    friend Formula Box(const Formula& lhs);
    friend Formula Box(const Group& g, const Formula& lhs);
	friend Formula Box(const GroupVar& x, const Formula& lhs);
    friend Formula Dia(const Formula& lhs);
    friend Formula Dia(const Group& g, const Formula& lhs);
	friend Formula Dia(const GroupVar& x, const Formula& lhs);
	friend Formula Would(const Formula& lhs, const Formula& rhs);
	friend Formula Would(const Group& g, const Formula& lhs, const Formula& rhs);
	friend Formula Might(const Formula& lhs, const Formula& rhs);
	friend Formula Might(const Group& g, const Formula& lhs, const Formula& rhs);
	friend Formula Ex(const qterm& condition, const Formula& lhs);
	friend Formula Fa(const qterm& condition, const Formula& lhs);

    public:
    Formula( const Atom& a ) : node( new BoolNode( a )) {};
	Formula( const bool& b ) {
		if( b )
			node = new TrueNode();
		else
			node = new FalseNode();
	};
    Formula( Node* n ) : node( n ) {};

	Formula subst(const Group& g, const GroupVar& x) const;  //g for x
	Formula subst(const GroupVar& y, const GroupVar& x) const;  //y for x
    inline std::string print()
        { return node->print(); }
	inline std::string print() const
        { return node->print(); }
    inline Node* getNode() const
        { return node->getNode(); }
	inline bool isTrueNode() const
		{ return ( typeid( *(node->getNode()) ) == typeid( TrueNode ) ); }
	inline bool isFalseNode() const
		{ return ( typeid( *(node->getNode()) ) == typeid( FalseNode ) ); }
	inline bool isLEQ() const
		{ return ( typeid( *(node->getNode()) ) == typeid( LEQNode ) ); }
	inline bool isGEQ() const
		{ return ( typeid( *(node->getNode()) ) == typeid( GEQNode ) ); }
	inline bool isNotLEQ() const
		{ return ( typeid( *(node->getNode()) ) == typeid( NotLEQNode ) ); }
	inline bool isNotGEQ() const
		{ return ( typeid( *(node->getNode()) ) == typeid( NotGEQNode ) ); }
	inline bool isOr() const
		{ return ( typeid( *(node->getNode()) ) == typeid( OrNode ) ); }
	inline bool isAnd() const
		{ return ( typeid( *(node->getNode()) ) == typeid( AndNode ) ); }
	inline bool isNot() const
		{ return ( typeid( *(node->getNode()) ) == typeid( NegatedNode ) ); }
	inline bool isBool() const
		{ return ( typeid( *(node->getNode()) ) == typeid( BoolNode ) ); }
	inline bool isLiteral() const
		{
			if( this->isBool() )
				return true;
			if( this->isNot() )
				return ( typeid( *( ((node->getNode()))->getChild()) ) == typeid( BoolNode ) );
			return false;
		}
	inline bool isDia() const
		{ return ( typeid( *(node->getNode()) ) == typeid( DiamondNode ) ); }
	inline bool isVarDia() const
		{ return ( typeid( *(node->getNode()) ) == typeid( DiamondVarNode ) ); }
	inline bool isBox() const
		{ return ( typeid( *(node->getNode()) ) == typeid( BoxNode ) ); }
	inline bool isVarBox() const
		{ return ( typeid( *(node->getNode()) ) == typeid( BoxVarNode ) ); }
	inline bool isExists() const
		{ return ( typeid( *(node->getNode()) ) == typeid( ExistsNode ) ); }
	inline bool isForall() const
		{ return ( typeid( *(node->getNode()) ) == typeid( ForallNode ) ); }
	inline bool isWould() const
		{ return ( typeid( *(node->getNode()) ) == typeid( WouldNode ) ); }
	inline bool isMight() const
		{ return ( typeid( *(node->getNode()) ) == typeid( MightNode ) ); }
	
private:
    Node* node;
};

Formula operator~(const Formula& lhs);

inline Formula LEQ( const GroupVar& x, const Group& g ) {
	return new LEQNode( x, g );
}

inline Formula LEQ( const Group& g, const GroupVar& x ) {
	return new GEQNode( x, g );
}

inline Formula GEQ( const GroupVar& x, const Group& g ) {
	return new GEQNode( x, g );
}

inline Formula NotLEQ( const GroupVar& x, const Group& g ) {
	return new NotLEQNode( x, g );
}

inline Formula NotLEQ( const Group& g, const GroupVar& x ) {
	return new NotGEQNode( x, g );
}

inline Formula NotGEQ( const GroupVar& x, const Group& g ) {
	return new NotGEQNode( x, g );
}

inline Formula operator&(const Formula& lhs, const Formula& rhs )
{
    return Formula( new AndNode( lhs.node, rhs.node ) );
}
inline Formula operator|(const Formula& lhs, const Formula& rhs )
{
    return Formula( new OrNode( lhs.node, rhs.node ) );
}
inline Formula Box(const Group& g, const Formula& lhs)
{
    return Formula( new BoxNode( lhs.node, g ) );
}

inline Formula Box(const GroupVar& x, const Formula& lhs)
{
    return Formula( new BoxVarNode( lhs.node, x ) );
}

inline Formula Dia(const Group& g, const Formula& lhs)
{
    return Formula( new DiamondNode( lhs.node, g ) );
}

inline Formula Dia(const GroupVar& x, const Formula& lhs)
{
    return Formula( new DiamondVarNode( lhs.node, x ) );
}

inline Formula Would(const Group& g, const Formula& lhs, const Formula& rhs)
{
	return Formula( new WouldNode( lhs.node, rhs.node, g ) );
}

inline Formula Might(const Group& g, const Formula& lhs, const Formula& rhs)
{
	return Formula( new MightNode( lhs.node, rhs.node, g ) );
}

inline Formula Ex(const qterm& condition, const Formula& lhs)
{
	return Formula( new ExistsNode( lhs.node, std::get<0>(condition), std::get<1>(condition), std::get<2>(condition) ) );
}

inline Formula Fa(const qterm& condition, const Formula& lhs)
{
    return Formula( new ForallNode( lhs.node, std::get<0>(condition), std::get<1>(condition), std::get<2>(condition) ) );
}



inline Group getLowerBound( const Formula& f ) {
	Node* n = f.getNode();
	if( typeid( *n ) == typeid( ExistsNode ) )
		return ((ExistsNode*)f.getNode())->getLowerBound();
	if( typeid( *n ) == typeid( ForallNode ) )
		return ((ForallNode*)f.getNode())->getLowerBound();

	assert(false);
}

inline Group getUpperBound( const Formula& f ) {
	Node* n = f.getNode();
	if( typeid( *n ) == typeid( ExistsNode ) )
		return ((ExistsNode*)f.getNode())->getUpperBound();
	if( typeid( *n ) == typeid( ForallNode ) )
		return ((ForallNode*)f.getNode())->getUpperBound();

	assert(false);
}

inline const Formula getChild( const Formula& f) {
	Node* n = f.getNode();
	return n->getChild();
}

inline const Formula getLHS( const Formula& f) {
	Node* n = f.getNode();
	return n->getLhsNode();
}

inline const Formula getRHS( const Formula& f) {
	Node* n = f.getNode();
	return n->getRhsNode();
}
inline Group getGroup( const Formula& f ) {
	Node* n = f.getNode();
	return n->getGroup();
}

inline GroupVar getVar( const Formula& f ) {
	Node* n = f.getNode();
	return n->getVar();
}


#endif /* NODES_HH */
